'use client';
import { useInventory } from '../providers/inventory-provider';
import { getProfitableReorderItems } from '@/lib/inventory-calculations';
import {
  BarChart,
  Bar,
  XAxis,
  YAxis,
  Tooltip,
  Legend,
} from 'recharts';
import { ChartContainer, ChartTooltipContent } from '../ui/chart';
import { CardDescription } from '../ui/card';

export function ReorderProfitabilityChart() {
  const { inventoryData } = useInventory();
  const profitableReorders = getProfitableReorderItems(inventoryData);

  const formatCurrency = (value: number) => {
    return new Intl.NumberFormat('en-IN', {
      style: 'currency',
      currency: 'INR',
      maximumFractionDigits: 0,
    }).format(value);
  };

  if (profitableReorders.length === 0) {
    return (
      <div className="flex flex-col items-center justify-center h-full text-muted-foreground p-8 text-center">
        <CardDescription>
            No profitable items need reordering based on current data.
        </CardDescription>
      </div>
    );
  }

  return (
    <div className="w-full">
      <CardDescription className="text-center mb-4 px-4">
        This chart shows items that are below their reorder level, prioritized by their potential profit.
      </CardDescription>
      <ChartContainer
        config={{
          profit: { label: 'Potential Profit', color: 'hsl(var(--chart-3))' },
        }}
        className="min-h-[400px] w-full"
      >
        <BarChart
          data={profitableReorders.slice(0, 10)} // Show top 10
          layout="vertical"
          margin={{ left: 100, right: 20, top: 20, bottom: 20 }}
          animationDuration={500}
          animationEasing="ease-in-out"
        >
          <XAxis type="number" tickFormatter={formatCurrency} />
          <YAxis
            type="category"
            dataKey="Item Name"
            width={100}
            tickLine={false}
            axisLine={false}
            interval={0}
            tick={{fontSize: 12}}
          />
          <Tooltip
            cursor={{ fill: 'hsl(var(--muted))' }}
            content={
              <ChartTooltipContent
                formatter={(value) =>
                  typeof value === 'number'
                    ? formatCurrency(value)
                    : value.toString()
                }
              />
            }
          />
          <Legend />
          <Bar
            dataKey="Product-wise Profit"
            name="Profit"
            fill="var(--color-profit)"
            radius={4}
          />
        </BarChart>
      </ChartContainer>
    </div>
  );
}
